"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _bl = _interopRequireDefault(require("bl"));
var _stream = require("stream");
var _message = _interopRequireDefault(require("./message"));
var _packet = require("./packet");
var _errors = require("./errors");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/**
  IncomingMessageStream
  Transform received TDS data into individual IncomingMessage streams.
*/
class IncomingMessageStream extends _stream.Transform {
  constructor(debug) {
    super({
      readableObjectMode: true
    });
    this.debug = debug;
    this.currentMessage = undefined;
    this.bl = new _bl.default();
  }
  pause() {
    super.pause();
    if (this.currentMessage) {
      this.currentMessage.pause();
    }
    return this;
  }
  resume() {
    super.resume();
    if (this.currentMessage) {
      this.currentMessage.resume();
    }
    return this;
  }
  processBufferedData(callback) {
    // The packet header is always 8 bytes of length.
    while (this.bl.length >= _packet.HEADER_LENGTH) {
      // Get the full packet length
      const length = this.bl.readUInt16BE(2);
      if (length < _packet.HEADER_LENGTH) {
        return callback(new _errors.ConnectionError('Unable to process incoming packet'));
      }
      if (this.bl.length >= length) {
        const data = this.bl.slice(0, length);
        this.bl.consume(length);

        // TODO: Get rid of creating `Packet` instances here.
        const packet = new _packet.Packet(data);
        this.debug.packet('Received', packet);
        this.debug.data(packet);
        let message = this.currentMessage;
        if (message === undefined) {
          this.currentMessage = message = new _message.default({
            type: packet.type(),
            resetConnection: false
          });
          this.push(message);
        }
        if (packet.isLast()) {
          // Wait until the current message was fully processed before we
          // continue processing any remaining messages.
          message.once('end', () => {
            this.currentMessage = undefined;
            this.processBufferedData(callback);
          });
          message.end(packet.data());
          return;
        } else if (!message.write(packet.data())) {
          // If too much data is buffering up in the
          // current message, wait for it to drain.
          message.once('drain', () => {
            this.processBufferedData(callback);
          });
          return;
        }
      } else {
        break;
      }
    }

    // Not enough data to read the next packet. Stop here and wait for
    // the next call to `_transform`.
    callback();
  }
  _transform(chunk, _encoding, callback) {
    this.bl.append(chunk);
    this.processBufferedData(callback);
  }
}
var _default = exports.default = IncomingMessageStream;
module.exports = IncomingMessageStream;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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